﻿' Simple Animal Blocks Game in VB.NET (Windows Forms)
' Create a new Windows Forms App project and replace Form1 code with this.

Public Class Form1
    Private score As Integer = 0
    Private animCluster As List(Of Point) = Nothing
    Private animTimer As New Timer() With {.Interval = 150}
    Private animState As Boolean = False
    Private grid(9, 9) As Integer ' 10x10 grid
    Private rand As New Random()
    Private cellSize As Integer = 40

    Private Sub Form1_Load(sender As Object, e As EventArgs) Handles MyBase.Load
        animTimer.Enabled = False
        AddHandler animTimer.Tick, AddressOf AnimTick
        InitGrid()
        UpdateScore()
        Me.Width = 420
        Me.Height = 442
        Me.StartPosition = FormStartPosition.CenterScreen
        Me.CenterToScreen()
        InitGrid()
    End Sub

    Private Sub InitGrid()
        For r = 0 To 9
            For c = 0 To 9
                grid(r, c) = rand.Next(0, 4) ' 4 animal types
            Next
        Next
        Me.Invalidate()
    End Sub

    Private Function ColorFromAnimal(a As Integer) As Color
        Select Case a
            Case 0 : Return Color.Red
            Case 1 : Return Color.Blue
            Case 2 : Return Color.Green
            Case 3 : Return Color.Yellow
            Case Else : Return Color.Gray
        End Select
    End Function

    Private Sub Form1_MouseClick(sender As Object, e As MouseEventArgs) Handles Me.MouseClick
        Dim c As Integer = e.X \ cellSize
        Dim r As Integer = e.Y \ cellSize
        If r < 0 Or r > 9 Or c < 0 Or c > 9 Then Exit Sub

        Dim target = grid(r, c)

        Dim visited(9, 9) As Boolean
        Dim cluster As New List(Of Point)
        FindCluster(r, c, target, visited, cluster)

        If cluster.Count > 1 Then
            animCluster = cluster
            animState = False
            animTimer.Start()
            Exit Sub
            For Each p In cluster
                grid(p.X, p.Y) = -1
            Next
            ApplyGravityAnimated()

            ' Combo bonus logic
            If (DateTime.Now - lastClearTime).TotalMilliseconds < 800 Then
                comboMultiplier += 1
            Else
                comboMultiplier = 1
            End If
            lastClearTime = DateTime.Now

            score += cluster.Count * 10 * comboMultiplier
            UpdateScore()
            Me.Invalidate()
        End If
    End Sub

    Private Sub FindCluster(r As Integer, c As Integer, target As Integer, visited(,) As Boolean, cluster As List(Of Point))
        If r < 0 Or r > 9 Or c < 0 Or c > 9 Then Exit Sub
        If visited(r, c) Or grid(r, c) <> target Then Exit Sub

        visited(r, c) = True
        cluster.Add(New Point(r, c))

        FindCluster(r - 1, c, target, visited, cluster)
        FindCluster(r + 1, c, target, visited, cluster)
        FindCluster(r, c - 1, target, visited, cluster)
        FindCluster(r, c + 1, target, visited, cluster)
    End Sub

    Private Sub ApplyGravityAnimated()
        ' Animated gravity: blocks fall one row at a time
        Dim moved As Boolean = True
        While moved
            moved = False
            For r = 8 To 0 Step -1
                For c = 0 To 9
                    If grid(r + 1, c) = -1 And grid(r, c) <> -1 Then
                        grid(r + 1, c) = grid(r, c)
                        grid(r, c) = -1
                        moved = True
                    End If
                Next
            Next
            Me.Invalidate()
            Threading.Thread.Sleep(40) ' small animation delay
        End While
    End Sub

    ' Combo bonus: increase multiplier if multiple clusters cleared quickly
    Private comboMultiplier As Integer = 1
    Private lastClearTime As DateTime = DateTime.MinValue

    Private Sub ApplyGravity() ' fallback not used()
        For c = 0 To 9
            Dim stack As New List(Of Integer)
            For r = 9 To 0 Step -1
                If grid(r, c) <> -1 Then
                    stack.Add(grid(r, c))
                End If
            Next
            While stack.Count < 10
                stack.Add(rand.Next(0, 4))
            End While
            For r = 9 To 0 Step -1
                grid(r, c) = stack(9 - r)
            Next
        Next
    End Sub
    Private Sub AnimTick(sender As Object, e As EventArgs)
        animState = Not animState
        If animState = False Then
            For Each p In animCluster
                grid(p.X, p.Y) = -1
            Next
            animTimer.Stop()
            ApplyGravity()
            score += animCluster.Count * 10
            UpdateScore()
            animCluster = Nothing
        End If
        Me.Invalidate()
    End Sub

    Private Sub UpdateScore()
        Me.Text = "Animal Blocks - Score: " & score
    End Sub

    ' ==== Combo Text Animation ====
    Private comboText As String = ""
    Private comboAlpha As Integer = 0
    Private comboY As Integer = 50
    Private comboTimer As New Timer() With {.Interval = 30}

    Private Sub ShowComboText()
        comboText = "Combo x" & comboMultiplier & "!"
        comboAlpha = 255
        comboY = 80
        AddHandler comboTimer.Tick, AddressOf ComboAnimTick
        comboTimer.Start()
    End Sub

    Private Sub ComboAnimTick(sender As Object, e As EventArgs)
        comboY -= 2 ' slide upward
        comboAlpha -= 6 ' fade out
        If comboAlpha <= 0 Then
            comboTimer.Stop()
            comboText = ""
        End If
        Me.Invalidate()
    End Sub

    ' Modify scoring to trigger combo text
    Private Sub TriggerComboDisplay()
        If comboMultiplier > 1 Then ShowComboText()
    End Sub

    ' Modify Paint to draw combo text with easing + ghost trail
    Private Sub Form1_Paint(sender As Object, e As PaintEventArgs) Handles Me.Paint
        For r = 0 To 9
            For c = 0 To 9
                Dim color As Color = ColorFromAnimal(grid(r, c))
                Using b As New SolidBrush(color)
                    e.Graphics.FillRectangle(b, c * cellSize, r * cellSize, cellSize - 2, cellSize - 2)
                End Using
            Next
        Next
        ' Existing grid drawing preserved above this

        If comboText <> "" Then
            Dim ghostOffset As Integer = 5
            Dim alphaGhost As Integer = Math.Max(comboAlpha - 120, 0)
            Using ghostBrush As New SolidBrush(Color.FromArgb(alphaGhost, Color.White))
                e.Graphics.DrawString(comboText, New Font("Arial", 24, FontStyle.Bold), ghostBrush, 100, comboY + ghostOffset)
            End Using

            Using mainBrush As New SolidBrush(Color.FromArgb(comboAlpha, Color.Yellow))
                e.Graphics.DrawString(comboText, New Font("Arial", 28, FontStyle.Bold), mainBrush, 100, comboY)
            End Using
        End If
    End Sub



    ' ===================== PARTICLE EFFECTS =====================
    Private particleList As New List(Of Particle)
    Private particleTimer As New Timer() With {.Interval = 30}

    Private Sub InitParticles()
        AddHandler particleTimer.Tick, AddressOf ParticleTick
        particleTimer.Start()
    End Sub

    Private Sub ParticleTick(sender As Object, e As EventArgs)
        For i = particleList.Count - 1 To 0 Step -1
            Dim p = particleList(i)
            p.X += p.Vx
            p.Y += p.Vy
            p.Life -= 10
            If p.Life <= 0 Then
                particleList.RemoveAt(i)
            End If
        Next
        Me.Invalidate()
    End Sub

    Private Sub SpawnClusterParticles(cluster As List(Of Point))
        For Each p In cluster
            For i = 1 To 12 ' number of sparkles per block
                Dim part As New Particle()
                part.X = p.Y * cellSize + cellSize \ 2
                part.Y = p.X * cellSize + cellSize \ 2
                Dim angle As Double = rand.NextDouble() * Math.PI * 2
                Dim speed As Double = rand.NextDouble() * 5 + 1
                part.Vx = Math.Cos(angle) * speed
                part.Vy = Math.Sin(angle) * speed
                part.Life = 255
                part.Color = Color.FromArgb(255, Color.White)
                particleList.Add(part)
            Next
        Next
    End Sub

    ' Modify existing cluster removal to include particles
    ' After: For Each p In animCluster
    ' Insert: SpawnClusterParticles(animCluster)

    ' Modify Paint to draw particles
    ' Add inside Paint:
    '   For Each part In particleList
    '       Using b As New SolidBrush(Color.FromArgb(part.Life, Color.Yellow))
    '           e.Graphics.FillEllipse(b, part.X, part.Y, 6, 6)
    '       End Using
    '   Next

End Class

' Particle object
Public Class Particle
    Public X As Double
    Public Y As Double
    Public Vx As Double
    Public Vy As Double
    Public Life As Integer
    Public Color As Color
End Class

